<?php

/**
 * Plugin Name: WooCommerce Smart Processing Solution Gateway
 * Plugin URI: https://smartprocessingsolution.com
 * Description: Digital Payment made simple and transparent
 * Developer: Smart Processing Solution
 * Developer URI: https://smartprocessingsolution.com/
 * Author: Smart Processing Solution
 * Author URI: https://smartprocessingsolution.com/
 * Version: 4.0.3
 *
 * Text Domain: woocommerce-gateway-sps
 * Domain Path: /i18n/languages/
 *
 * Requires at least: 4.2
 * Tested up to: 6.6
 *
 * Copyright: © 2009-2024 Smart Processing Solution / Smart Processing Solution
 * License: GNU General Public License v3.0
 * License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WC SPS Payment gateway plugin class.
 *
 * @class WC_SPS_Payments
 */
class WC_SPS_Payments {

	/**
	 * Plugin bootstrapping.
	 */
	public static function init() {

		// SPS Payments gateway class.
		add_action( 'plugins_loaded', array( __CLASS__, 'includes' ), 0 );

		// Make the SPS Payments gateway available to WC.
		add_filter( 'woocommerce_payment_gateways', array( __CLASS__, 'add_gateway' ) );

		// Registers WooCommerce Blocks integration.
		add_action( 'woocommerce_blocks_loaded', array( __CLASS__, 'woocommerce_gateway_sps_woocommerce_block_support' ) );

        // Add a setting link to the plugin block in the admin
        add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( __CLASS__, 'woocommerce_gateway_sps_plugin_links') );

        add_filter( 'woocommerce_add_to_cart_validation', array( __CLASS__, 'woocommerce_gateway_sps_add_to_cart_validation'), 10, 2 );
	}

	/**
	 * Add the SPS Payment gateway to the list of available gateways.
	 *
	 * @param array $gateways Array of payment gateway instances
	 * @return array Modified array of payment gateway instances
	 */
	public static function add_gateway($gateways) {
		$gateways[] = 'WC_Gateway_SPS';
		return $gateways;
	}

	/**
	 * Plugin includes.
	 */
	public static function includes() {

		// Make the WC_Gateway_SPS class available.
		if ( class_exists( 'WC_Payment_Gateway' ) ) {
			require_once 'includes/class-wc-sps-api-client.php';
			require_once 'includes/class-wc-gateway-sps.php';
		}
	}

	/**
	 * Plugin url.
	 *
	 * @return string
	 */
	public static function plugin_url() {
		return untrailingslashit( plugins_url( '/', __FILE__ ) );
	}

	/**
	 * Plugin url.
	 *
	 * @return string
	 */
	public static function plugin_abspath() {
		return trailingslashit( plugin_dir_path( __FILE__ ) );
	}

	/**
	 * Registers WooCommerce Blocks integration.
	 *
	 */
	public static function woocommerce_gateway_sps_woocommerce_block_support() {
		if ( class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
			require_once 'includes/blocks/class-wc-sps-payments-blocks.php';
			add_action(
				'woocommerce_blocks_payment_method_type_registration',
				function( Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry ) {
					$payment_method_registry->register( new WC_Gateway_SPS_Blocks_Support() );
				}
			);
		}
	}

    public static function woocommerce_gateway_sps_plugin_links( $links ): array
    {
        $settings_url = add_query_arg(
            array(
                'page' => 'wc-settings',
                'tab' => 'checkout',
                'section' => 'sps',
            ),
            admin_url( 'admin.php' )
        );

        $plugin_links = array(
            '<a href="' . esc_url( $settings_url ) . '">' . __( 'Settings', 'woocommerce-gateway-sps' ) . '</a>',
            '<a href="https://smartprocessingsolution.com/">' . __( 'Support', 'woocommerce-gateway-sps' ) . '</a>'
        );

        return array_merge( $plugin_links, $links );
    }

    public static function woocommerce_gateway_sps_add_to_cart_validation( $passed_validation, $product_id)
    {
        $product = wc_get_product( $product_id );
        if (!$product) {
            return $passed_validation;
        }

        $type = $product->get_type();
        if ($type == 'subscription' || $type == 'variable-subscription' || $type == 'subscription_variation') {
            // Fix product meta
            if ( ! $product->get_sold_individually() ) {
                $product->set_sold_individually( true );
                $product->save();
            }
            // Check functionalities
            $trial = $product->get_meta('_subscription_trial_length');
            if (!empty($trial) && $trial > 0) {
                wc_add_notice( __( 'Sorry, the trial period is not currently supported by SPS on subscription. Please disable it for this product.', 'woocommerce-gateway-sps' ), 'error' );
                return false;
            }

            if (WC()->cart->is_empty())
                return true;

            wc_add_notice( __( 'You cannot add a subscription product to a cart with other items.', 'woocommerce-gateway-sps' ), 'error' );
            return false;
        }

        // Check if the cart contains any subscriptions
        foreach ( WC()->cart->get_cart() as $cart_item_key => $cart_item ) {
            $cart_product = wc_get_product( $cart_item['product_id'] );
            if ( $cart_product && in_array($cart_product->get_type(), ['subscription', 'variable-subscription', 'subscription_variation']) ) {
                wc_add_notice( __( 'You can only have one subscription product in your cart.', 'woocommerce-gateway-sps' ), 'error' );
                return false;
            }
        }

        return $passed_validation;
    }
}

WC_SPS_Payments::init();
