<?php
/**
 * SPS API Client
 *
 * Handles all API interactions with the SPS payment gateway.
 *
 * @package WC_SPS_Payments
 */

defined('ABSPATH') || exit;

/**
 * WC_SPS_API_Client class.
 */
class WC_SPS_API_Client {
    /**
     * API Endpoints
     */
    const DEV_API_URL = 'https://www.smartprocessingsolution.ovh/api/processing';
    const SANDBOX_API_URL = 'https://www.smartprocessingsolution.biz/api/processing';
    const PRODUCTION_API_URL = 'https://www.smartprocessingsolution.com/api/processing';

    /**
     * API Authentication token
     *
     * @var string
     */
    private $api_token;

    /**
     * Whether to use sandbox mode
     *
     * @var bool
     */
    private $is_sandbox;

    /**
     * Constructor
     *
     * @param string $api_token API Token
     * @param bool   $is_sandbox Whether to use sandbox mode
     */
    public function __construct($api_token, $is_sandbox = false) {
        $this->api_token = $api_token;
        $this->is_sandbox = $is_sandbox;
    }

    /**
     * Get base API URL based on environment
     *
     * @return string
     */
    private function get_api_url() {
        return $this->is_sandbox ? self::SANDBOX_API_URL : self::PRODUCTION_API_URL;
    }

    /**
     * Make an API request
     *
     * @param string $endpoint API endpoint
     * @param array  $args Request arguments
     * @param string $method HTTP method
     * @return array|WP_Error
     */
    private function request($endpoint, $args = [], $method = 'GET') {
        $url = trailingslashit($this->get_api_url()) . ltrim($endpoint, '/');

        $headers = [
            'Authorization' => 'Bearer ' . $this->api_token,
            'Content-Type' => 'application/json',
        ];

        $request_args = [
            'method'  => $method,
            'headers' => $headers,
            'timeout' => 30,
        ];

        if (!empty($args) && in_array($method, ['POST', 'PUT'])) {
            $request_args['body'] = wp_json_encode($args);
        }

        if (!empty($args) && $method === 'GET') {
            $url = add_query_arg($args, $url);
        }

        $response = wp_remote_request($url, $request_args);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $code = wp_remote_retrieve_response_code($response);

        if ($code >= 400) {
            return new WP_Error(
                'sps_api_error',
                sprintf(__('API Error: %s', 'woocommerce-gateway-sps'), $body),
                ['status' => $code]
            );
        }

        return json_decode($body, true);
    }

    /**
     * Create a payment request
     *
     * @param array $payment_request Payment request data
     * @param array $customer Customer data
     * @return array|WP_Error
     */
    public function create_payment_request($payment_request, $customer) {
        return $this->request('payment-request/create', [
            'payment_request' => $payment_request,
            'customer' => $customer,
        ], 'POST');
    }

    /**
     * Get a payment request
     *
     * @param int $id Payment request ID
     * @return array|WP_Error
     */
    public function get_payment_request($id) {
        return $this->request("payment-request/get/{$id}");
    }

    /**
     * Cancel a payment request
     *
     * @param int $id Payment request ID
     * @param string $reason Cancellation reason
     * @return array|WP_Error
     */
    public function cancel_payment_request($id, $reason = '') {
        return $this->request("payment-request/cancel/{$id}", [
            'reason' => $reason,
        ]);
    }

    /**
     * Get a request/transaction
     *
     * @param int $id Request ID
     * @return array|WP_Error
     */
    public function get_request($id) {
        return $this->request("request/get/{$id}");
    }

    /**
     * Search requests/transactions by order_id
     *
     * @param int $order_id Order ID
     * @return array|WP_Error
     */
    public function search_requests($criterias = []) {
        // Example of criterias:
        // [
        //     'reference' => '1234',
        //     'status' => 'completed'
        // ]
        return $this->request("requests/search", $criterias, "POST");
    }

    /**
     * Refund a request/transaction
     *
     * @param int    $id Request ID
     * @param string $secret_key Secret key for refunds
     * @param string $reason Refund reason
     * @param float  $amount Refund amount (optional)
     * @param string $currency Currency code (required if amount is specified)
     * @return array|WP_Error
     */
    public function refund_request($id, $secret_key, $reason, $amount = null, $currency = null) {
        $args = [
            'id' => $id,
            'secret_key' => $secret_key,
            'reason' => $reason,
        ];

        if ($amount !== null) {
            $args['amount'] = $amount;
            $args['currency'] = $currency;
        }

        return $this->request('request/refund', $args, 'POST');
    }

    /**
     * Get customer details
     *
     * @param int|string $id_or_email Customer ID or email
     * @return array|WP_Error
     */
    public function get_customer($id_or_email) {
        $args = is_email($id_or_email) ? ['email' => $id_or_email] : ['id' => $id_or_email];
        return $this->request('customer/get', $args);
    }
} 
