<?php

/**
 * WooCommerce SPS Payments Gateway
 *
 * Provides access to the SPS Payments Gateway
 *
 * @category    Payment Gateways
 * @author      Smart Processing Solution
 * @package     WooCommerce SPS Payments Gateway
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SPS Gateway.
 *
 * @class    WC_Gateway_SPS
 * @version  1.10.0
 */
class WC_Gateway_SPS extends WC_Payment_Gateway {

    public $id = 'sps';
    public string $version = '4.0.3';
    private array $available_currencies;
    private string $merchant_id;
    private string $sub_account;
    private string $merchant_key;
    private string $url;
    private string $response_url;
    private string $complete_status;
    private bool $send_debug_email;
    private bool $process_refund_ipn;
    protected string $instructions;
    protected bool $hide_for_non_admin_users;

	/**
	 * Constructor for the gateway.
	 */
	public function __construct() {
		
		$this->icon = apply_filters( 'woocommerce_sps_gateway_icon', 'https://www.smartprocessingsolution.com/files/smartprocessing/woocommerce.png' );
		$this->has_fields = false;

        $this->method_title = _x( 'SPS Payment', 'SPS payment method', 'woocommerce-gateway-sps' );
        $this->method_description = __( 'The WooCommerce Plugin operates by directing the customer to the SPS Gateway, where they can pay with their Debit or Credit Card.', 'woocommerce-gateway-sps' );

        // Setup available currency codes.
        $this->available_currencies = array( 'USD' );

        // Load the form fields.
        $this->init_form_fields();

        // Load the settings.
        $this->init_settings();

        // Setup constants.
        if ( ! is_admin() ) {
            $this->setup_constants();
        }

		$this->supports = array(
			//'pre-orders',
			'products',
			'subscriptions',
			'refunds',
			/*'subscription_cancellation',
			'subscription_suspension',
			'subscription_reactivation',
			'subscription_amount_changes',
			'subscription_date_changes',
			'multiple_subscriptions'*/
		);

        // Setup default merchant data.
        $this->merchant_id = $this->get_option( 'merchant_id' );
        $this->sub_account = $this->get_option( 'sub_account' );
        $this->merchant_key = $this->get_option( 'merchant_key' );
        $this->complete_status = $this->get_option( 'complete_status' );
        $this->process_refund_ipn = 'yes' === $this->get_option( 'process_refund_ipn' );
        $this->url = 'https://www.smartprocessingsolution.com/gateway/';
        $this->title = $this->get_option( 'title' );
        $this->description = $this->get_option( 'description' );
        $this->instructions = $this->get_option( 'instructions', $this->description );
        $this->response_url = WC()->api_request_url('WC_Gateway_SPS');
        $this->send_debug_email = 'yes' === $this->get_option( 'send_debug_email' );
		$this->hide_for_non_admin_users = $this->get_option( 'hide_for_non_admin_users' );

        // Set up the test data, if in test mode.
        if ( 'yes' === $this->get_option( 'test_mode' ) ) {
            $base_url = $this->get_option( 'test_url' );
            if (empty($base_url))
                $base_url = 'https://www.smartprocessingsolution.biz';
            $this->url = $base_url . '/gateway/';
            $this->add_test_mode_admin_settings_notice();
        } else {
            $this->send_debug_email = false;
        }

		// Actions.
		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		//add_action( 'woocommerce_scheduled_subscription_payment_sps', array( $this, 'process_subscription_payment' ), 10, 2 );
		//add_action ( 'wc_pre_orders_process_pre_order_completion_payment_' . $this->id, array( $this, 'process_pre_order_release_payment' ), 10 );
        add_action( 'woocommerce_api_wc_gateway_' . $this->id, array( $this, 'check_api_request' ) );
        add_action( 'woocommerce_receipt_' . $this->id, array( $this, 'receipt_page' ) );

        // Check if the base currency supports this gateway.
        if ( ! $this->is_valid_for_use() ) {
            $this->enabled = false;
        }

        // Add is_available check for non-admin users
        add_filter('woocommerce_available_payment_gateways', array($this, 'maybe_hide_gateway_for_non_admin'));

        // Add admin notice for missing API credentials
        add_action('admin_notices', array($this, 'check_api_credentials'));
        add_action('admin_notices', array($this, 'check_version'));

        // Add payment verification hooks
        add_action('woocommerce_before_cancel_unpaid_order', array($this, 'verify_before_cancel'), 10, 1);
        add_action('woocommerce_order_status_pending', array($this, 'schedule_payment_check'), 10, 1);
        add_action('wc_sps_verify_payment', array($this, 'verify_payment_status'), 10, 1);
        add_action('template_redirect', array($this, 'check_return_payment_status'));

        // Add meta box for SPS transactions in order page
        add_action('add_meta_boxes', array($this, 'add_sps_transactions_meta_box'));

        // Add refund status change handler
        add_action('woocommerce_order_status_refunded', array($this, 'handle_manual_refund'), 10, 2);
	}

	/**
	 * Initialise Gateway Settings Form Fields.
	 */
	public function init_form_fields() {

		$this->form_fields = array(
			'enabled' => array(
				'title'   => __( 'Enable/Disable', 'woocommerce-gateway-sps' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable SPS Payments', 'woocommerce-gateway-sps' ),
				'default' => 'yes',
			),
			'hide_for_non_admin_users' => array(
				'type'    => 'checkbox',
				'label'   => __( 'Hide at checkout for non-admin users', 'woocommerce-gateway-sps' ),
				'default' => 'no',
			),
			'title' => array(
				'title'       => __( 'Title', 'woocommerce-gateway-sps' ),
				'type'        => 'text',
				'description' => __( 'This controls the title which the user sees during checkout.', 'woocommerce-gateway-sps' ),
				'default'     => _x( 'SPS Payment', 'SPS payment method', 'woocommerce-gateway-sps' ),
				'desc_tip'    => true,
			),
			'description' => array(
				'title'       => __( 'Description', 'woocommerce-gateway-sps' ),
				'type'        => 'textarea',
				'description' => __( 'Payment method description that the customer will see on your checkout.', 'woocommerce-gateway-sps' ),
				'default'     => __( 'Pay securely by debit/credit cards with SPS.', 'woocommerce-gateway-sps' ),
				'desc_tip'    => true,
			),
            'test_mode' => array(
                'title'       => __( 'Sandbox/test mode', 'woocommerce-gateway-sps' ),
                'type'        => 'checkbox',
                'description' => __( 'Place the payment gateway in sandbox/test mode.', 'woocommerce-gateway-sps' ),
                'default'     => 'no'
            ),
            'test_url' => array(
                'title'       => __( 'Sandbox URL', 'woocommerce-gateway-sps' ),
                'type'        => 'text',
                'description' => __( 'URL of the Smart Processing Solution sandbox (default: https://www.smartprocessingsolution.biz; do not change this value unless you are requested to do so by Smart Processing Solution).', 'woocommerce-gateway-sps' ),
                'default'     => 'https://www.smartprocessingsolution.biz'
            ),
            'merchant_id' => array(
                'title'       => __( 'Merchant ID', 'woocommerce-gateway-sps' ),
                'type'        => 'text',
                'description' => __( 'This is the merchant ID, received from Smart Processing Solution.', 'woocommerce-gateway-sps' ),
                'default'     => ''
            ),
            'sub_account' => array(
                'title'       => __( 'Sub-account', 'woocommerce-gateway-sps' ),
                'type'        => 'text',
                'description' => __( 'Enter a sub-account ID here if you wish to associate transactions from this site with a specific sub-account.', 'woocommerce-gateway-sps' ),
                'default'     => ''
            ),
            'merchant_key' => array(
                'title'       => __( 'Encryption / Hash Key', 'woocommerce-gateway-sps' ),
                'type'        => 'text',
                'description' => __( 'This is the merchant\'s encryption key, defined in the Smart Processing Solution merchant account, under "<a href="https://www.smartprocessingsolution.com/merchants/myaccount/settings#transactions-notifications" target="_blank" rel="noopener noreferrer">Account Settings > Credentials and security > Encryption/Hash Key</a>".', 'woocommerce-gateway-sps' ),
                'default'     => ''
            ),
            'sps_api_key' => array(
                'title'       => __( 'API Token', 'woocommerce-gateway-sps' ),
                'type'        => 'text',
                'description' => __( 'This is the merchant\'s API key, defined in the Smart Processing Solution merchant account, under "<a href="https://www.smartprocessingsolution.com/merchants/myaccount/settings#sps-api" target="_blank" rel="noopener noreferrer">Account Settings > Credentials and security > API Token</a>".', 'woocommerce-gateway-sps' ),
                'default'     => ''
            ),
            'sps_secret_key' => array(
                'title'       => __( 'Secret Key', 'woocommerce-gateway-sps' ),
                'type'        => 'text',
                'description' => __( 'This is the merchant\'s Secret Key, defined in the Smart Processing Solution merchant account, under "<a href="https://www.smartprocessingsolution.com/merchants/myaccount/settings#sps-api" target="_blank" rel="noopener noreferrer">Account Settings > Credentials and security >Secret Key</a>".', 'woocommerce-gateway-sps' ),
                'default'     => ''
            ),
            'complete_status' => array(
                'title'       => __( 'Order status after payment', 'woocommerce-gateway-sps' ),
                'type'        => 'select',
                'description' => __( 'Choose your desired order status after a successful payment.', 'woocommerce-gateway-sps' ),
                'default'     => 'processing',
                'options'     => [
                    'processing' => __( 'Processing (recommended)', 'woocommerce-gateway-sps' ),
                    'on-hold' => __( 'On-hold', 'woocommerce-gateway-sps' ),
                ]
            ),
            'send_debug_email' => array(
                'title'   => __( 'Send an alert on IPN processing error?', 'woocommerce-gateway-sps' ),
                'type'    => 'checkbox',
                'label'   => __( 'Send an alert to the provided e-mail address if the IPN fails to be processed.', 'woocommerce-gateway-sps' ),
                'default' => 'yes'
            ),
            'debug_email' => array(
                'title'       => __( 'Who receives the IPN alert?', 'woocommerce-gateway-sps' ),
                'type'        => 'text',
                'description' => __('The e-mail address to which the IPN error alert is sent.', 'woocommerce-gateway-sps' ),
                'default'     => get_option( 'admin_email' )
            ),
            'process_refund_ipn' => array(
                'title'   => __( 'Process refund automatically?', 'woocommerce-gateway-sps' ),
                'type'    => 'checkbox',
                'label'   => __( 'Automatically set the order as refunded when a refund IPN is received.', 'woocommerce-gateway-sps' ),
                'default' => 'yes'
            ),
            'refund_with_api' => array(
                'title'   => __( 'Refund with API?', 'woocommerce-gateway-sps' ),
                'type'    => 'checkbox',
                'label'   => __( 'Refund the transaction with the API when the order is refunded in WooCommerce (requires API key and Secret Key).', 'woocommerce-gateway-sps' ),
                'default' => 'yes'
            ),
            'verbose_logging' => array(
                'title'       => __( 'Verbose logging', 'woocommerce-gateway-sps' ),
                'type'        => 'checkbox',
                'description' => __( 'Enable verbose logging (mainly for debugging purposes, in case of transaction issues).', 'woocommerce-gateway-sps' ),
                'default'     => 'no'
            ),
            'validate_ipn_ip' => array(
                'title'       => __( 'Validate IPN IP', 'woocommerce-gateway-sps' ),
                'type'        => 'checkbox',
                'label'       => __( 'Validate that IPN requests come from SPS servers', 'woocommerce-gateway-sps' ),
                'description' => __( 'Enable this to verify that IPN notifications come from authorized SPS IP addresses. <span style="color: red;"><b>Disable only if you experience issues with IPN processing</b></span>.', 'woocommerce-gateway-sps' ),
                'default'     => 'yes'
            ),
            'validate_ipn_signature' => array(
                'title'       => __( 'Validate IPN Signature', 'woocommerce-gateway-sps' ),
                'type'        => 'checkbox',
                'label'       => __( 'Validate the signature of IPN notifications', 'woocommerce-gateway-sps' ),
                'description' => __( 'Enable this to verify the authenticity of IPN notifications using a hash signature. <span style="color: red;"><b>Disable only if you experience issues with IPN processing</b></span>.', 'woocommerce-gateway-sps' ),
                'default'     => 'yes'
            ),
		);
	}

    /**
     * add_test_mode_admin_settings_notice()
     * Add a notice to the merchant_key and merchant_id fields when in test mode.
     *
     * @since 1.1.0
     */
    public function add_test_mode_admin_settings_notice(): void
    {
        $this->form_fields['test_mode']['description']  .= '<br/><span style="color: red; font-weight: bold;">' . __( 'Test mode is currently enabled: no real transactions will be processed!', 'woocommerce-gateway-sps' ) . '.</span>';
        $this->form_fields['merchant_id']['description']  .= '<br/><span style="color: red; font-weight: bold;">' . __( 'Sandbox merchant ID currently in use', 'woocommerce-gateway-sps' ) . ' (' . esc_html( $this->merchant_id ) . ').</span>';
        $this->form_fields['merchant_key']['description'] .= '<br/><span style="color: red; font-weight: bold;">' . __( 'Sandbox encryption key currently in use', 'woocommerce-gateway-sps' ) . ' (' . esc_html( $this->merchant_key ) . ').</span>';
    }

    /**
     * is_valid_for_use()
     *
     * Check if this gateway is enabled and available in the base currency being traded with.
     *
     * @since 1.1.0
     */
    public function is_valid_for_use(): bool
    {
        $is_available          = false;
        $is_available_currency = in_array( get_woocommerce_currency(), $this->available_currencies );

        if ( $is_available_currency && $this->merchant_id && $this->merchant_key ) {
            $is_available = true;
        }
        return $is_available;
    }

    /**
     * Admin Panel Options
     * - Options for bits like 'title' and availability on a country-by-country basis
     *
     * @since 1.1.0
     */
    public function admin_options(): void
    {
        ?>
        <h2>
            <?php _e( 'Smart Processing Solution Gateway', 'woocommerce-gateway-sps' ); ?>
            <?php
            if ( function_exists( 'wc_back_link' ) ) {
                wc_back_link( __( 'Return to payments', 'woocommerce-gateway-sps' ), admin_url( 'admin.php?page=wc-settings&tab=checkout' ) );
            }
            ?>
        </h2>
        <?php
        if ( in_array( get_woocommerce_currency(), $this->available_currencies ) ) {
            ?>
                <div class="inline notice">
            <p>
                <strong><?php /** @noinspection HtmlUnknownTarget */
                    printf( __( 'To verify transactions and automatically process orders payments, set your notification URL <a href="%1$s" target="_blank" rel="noopener noreferrer">here</a> to the URL below<span style="color: red"><pre><code>%2$s</code></pre></span>', 'woocommerce-gateway-sps' ), 'https://smartprocessingsolution.com/merchants/', $this->response_url ); ?></strong>
            </p>
                </div>
            <?php
            parent::admin_options();
        } else {
            ?>
            <div class="inline error"><p><strong><?php _e( 'Gateway Disabled', 'woocommerce-gateway-sps' ); ?></strong> <?php echo sprintf( __( 'Choose USD as your store currency in %1$sPricing Options%2$s to enable the Smart Processing Solution Gateway.', 'woocommerce-gateway-sps' ), '<a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=general' ) ) . '">', '</a>' ); ?></p></div>
            <?php
        }
    }

    /**
     * Generate the Smart Processing Solution button link.
     *
     * @since 1.1.0
     */
    public function generate_smartprocessing_form( $order_id ): string
    {
        $order = wc_get_order( $order_id );

        // Construct variables for post
        $data_to_send = array(
            // Merchant details
            'merchant_id'   => $this->merchant_id,
            'sub_account_id'=> $this->sub_account,
            'completed_url' => $this->get_return_url( $order ),
            'cancelled_url' => $order->get_cancel_order_url(),
            'rejected_url'  => $order->get_cancel_order_url(),

            // Version information
            'platform' => array(
                'name' => 'WordPress',
                'wp_version' => get_bloginfo('version'),
                'wc_version' => WC()->version,
                'plugin_version' => $this->version,
            ),

            // Billing details
            'user' => array(
                'firstname' => $order->get_billing_first_name() ?? $order->get_shipping_first_name(),
                'lastname'  => $order->get_billing_last_name() ?? $order->get_shipping_last_name(),
                'address1'  => $order->get_billing_address_1() ?? $order->get_shipping_address_1(),
                'address2'  => $order->get_billing_address_2() ?? $order->get_shipping_address_2(),
                'zipcode'   => $order->get_billing_postcode() ?? $order->get_shipping_postcode(),
                'city'      => $order->get_billing_city() ?? $order->get_shipping_city(),
                'country'   => $order->get_billing_country() ?? $order->get_shipping_country(),
                'state'     => $order->get_billing_state() ?? $order->get_shipping_state(),
                'phone'     => $order->get_billing_phone() ?? $order->get_shipping_phone(),
                'email'     => $order->get_billing_email(),
            ),

            // Item details
            'order_ref'         => $order->get_id(),
            'amount'            => $order->get_total(),
            'currency'          => $order->get_currency(),
            'order_description' => sprintf( __( 'Order %s on %s', 'woocommerce-gateway-sps' ), $order->get_order_number(), get_bloginfo( 'name' ) ),
            'addinfo'           => $order->get_order_key(),
        );

        // Calculate the hash
        $data_to_send['hash'] = md5($this->merchant_id. $data_to_send['amount']. $data_to_send['currency']. $data_to_send['order_ref'].$this->merchant_key);

        /**
         * Check for subscription in the cart
         * Note: we can have only ONE subscription per order (checked in woocommerce_add_to_cart_validation)
         * @since 2.0.0
         */
        $periodicity_map = [
            'day' => 'days',
            'week' => 'weeks',
            'month' => 'months',
            'year' => 'years',
        ];
        $order_items = $order->get_items();
        foreach ( $order_items as $order_item ) {
            if (method_exists($order_item, 'get_product')) {
                $product = $order_item->get_product();
                if ( $product && in_array($product->get_type(), ['subscription', 'variable-subscription', 'subscription_variation']) ) {
                    $data_to_send['recurring'] = [
                        'enabled' => 'yes',
                        'amount'  => $product->get_meta('_subscription_price'),
                        'periodicity'  => $periodicity_map[$product->get_meta('_subscription_period')],
                        'cycles'  => $product->get_meta('_subscription_length'),
                    ];
                    // Calculate the start date
                    $ts = wcs_add_time(1, $product->get_meta('_subscription_period'), time());
                    $data_to_send['recurring']['start_date'] = date('Y-m-d', $ts);
                    $data_to_send['recurring']['hash'] = md5($data_to_send['recurring']['amount'] . $data_to_send['recurring']['cycles'] . $data_to_send['recurring']['periodicity'] . $data_to_send['recurring']['start_date'] . $this->merchant_key);
                }
            }
        }

        $smartprocessing_args_array = array();

        $this->log(json_encode($data_to_send), true);

        foreach($data_to_send as $key => $value) {
            if (is_array($value)) {
                foreach( $value as $k => $v)
                    $smartprocessing_args_array[] = '<input type="hidden" name="' . $key .'['.$k.']" value="' . esc_attr( $v ) . '" />';
            } else {
                $smartprocessing_args_array[] = '<input type="hidden" name="' . esc_attr( $key ) .'" value="' . esc_attr( $value ) . '" />';
            }
        }

        return '<form action="' . esc_url( $this->url ) . '" method="post" id="smartprocessing_payment_form">
				' . implode( '', $smartprocessing_args_array ) . '
				<input type="submit" class="button-alt" id="submit_smartprocessing_payment_form" value="' . __( 'Pay via SPS', 'woocommerce-gateway-sps' ) . '" /> <a class="button cancel" href="' . $order->get_cancel_order_url() . '">' . __( 'Cancel order &amp; restore cart', 'woocommerce-gateway-sps' ) . '</a>
				<script type="text/javascript">
					jQuery(function(){
						jQuery("body").block(
							{
								message: "' . __( 'Please wait, you will be redirected to the SPS gateway in order to make your payment.', 'woocommerce-gateway-sps' ) . '",
								overlayCSS:
								{
									background: "#fff",
									opacity: 0.6
								},
								css: {
							        padding:        20,
							        textAlign:      "center",
							        color:          "#555",
							        border:         "3px solid #aaa",
							        backgroundColor:"#fff",
							        cursor:         "wait"
							    }
							});
						jQuery( "#submit_smartprocessing_payment_form" ).click();
					});
				</script>
			</form>';
    }
    
	/**
	 * Process the payment and return the result.
	 *
	 * @param  int  $order_id
	 * @return array
	 */
	public function process_payment( $order_id ) {

        $order = wc_get_order( $order_id );
        $order->update_status('pending-payment', __('Awaiting payment', 'woocommerce'));
        return array(
            'result' 	 => 'success',
            'redirect'	 => $order->get_checkout_payment_url( true )
        );
	}


    /**
     * Reciept page.
     *
     * Display text and a button to direct the user to Smart Processing Solution.
     *
     * @since 1.1.0
     */
    public function receipt_page( $order ): void
    {
        echo '<p>' . __( 'Thank you for your order, please click the button below to pay with SPS.', 'woocommerce-gateway-sps' ) . '</p>';
        echo $this->generate_smartprocessing_form( $order );
    }

    /**
     * Check Smart Processing Solution IPN response.
     */
    public function check_api_request(): void {
        // Send headers early to ensure they are set
        header('Content-Type: application/json');
        $request = stripslashes_deep($_POST);
        
        if (!empty($request)) {
            if (!empty($request['action'])) {
                $response = $this->handle_api_request($request);
            } else if (!empty($request)) {
                $response = $this->handle_ipn_request($request);
            } else {
                $response = [
                    'status' => 'error',
                    'message' => 'Invalid action',
                ];
            }
        } else {
            $response = [
                'status' => 'error',
                'message' => 'No data received',
                'platform' => array(
                    'wp_version' => get_bloginfo('version'),
                    'wc_version' => WC()->version,
                    'plugin_version' => $this->version,
                ),
            ];
        }
        
        echo wp_json_encode($response);
        exit;
    }

    /**
     * Check Smart Processing Solution API request validity.
     *
     * @param array $data
     * @return array Response data
     */
    public function handle_api_request(array $data): array {

        if (empty($data['merchant_key']) || $data['merchant_key'] !== $this->merchant_key || empty($data['merchant_id']) || $data['merchant_id'] !== $this->merchant_id) {
            return [
                'status' => 'error',
                'message' => 'Invalid merchant credentials',
            ];
        }

        if (!empty($data['action']) && $data['action'] === 'get_settings') {
            $response = [
                'status' => 'ok',
                'message' => 'API request processed successfully',
                'settings' => $this->get_settings(),
            ];
        } else if (!empty($data['action']) && $data['action'] === 'get_order' && !empty($data['order_id'])) {
            $response = [
                'status' => 'ok',
                'message' => 'API request processed successfully',
                'order' => $this->get_order($data['order_id']),
            ];
        } else if (!empty($data['action']) && $data['action'] === 'get_logs') {
            $response = [
                'status' => 'ok',
                'message' => 'API request processed successfully',
                'logs' => $this->get_logs(),
            ];
        } else {
            $response = [
                'status' => 'error',
                'message' => 'Invalid action',
            ];
        }
        return $response;
    }

    /**
     * Check Smart Processing Solution IPN validity.
     *
     * @param array $data
     * @return array Response data
     */
    public function handle_ipn_request(array $data): array {
        $response = [
            'status' => 'ok',
            'message' => 'IPN processed successfully'
        ];

        $epError        = false;
        $epDone         = false;
        $processRefund  = false;
        $epDebugEmail   = $this->get_option( 'debug_email', get_option( 'admin_email' ) );
        $vendor_name    = get_bloginfo( 'name' );
        $vendor_url     = home_url( '/' );

        $this->log( "\n" . '----------' . "\n" . 'Smart Processing Solution IPN call received' );

        // Get data sent by Smart Processing Solution
        $this->log( 'Get posted data', true );
        $this->log( 'Smart Processing Solution Data: '. json_encode( $data ), true );

        if ( empty($data) ) {
            $epError  = true;
            $epErrMsg = SPS_ERR_BAD_ACCESS;
        }

        // Verify source IP
        if (! $epError) {
            $this->log( 'Verify source IP', true );
            if ('yes' === $this->get_option('validate_ipn_ip') && !$this->validate_ip($_SERVER['REMOTE_ADDR'])) {
                $epError  = true;
                $epErrMsg = SPS_ERR_BAD_SOURCE_IP;
            }
        }

        // Verify data received
        if( ! $epError ) {
            $this->log( 'Verify data received', true );
            $epValid = $this->validate_response_data( $data );
            if( ! $epValid ) {
                $epError = true;
                $epErrMsg = SPS_ERR_BAD_ACCESS;
            }
        }

        // Verify security signature
        if (! $epError) {
            $this->log( 'Verify security signature', true );
            // Signature verification failed
            if ('yes' === $this->get_option('validate_ipn_signature') && !$this->validate_signature($data)) {
                $epError  = true;
                $epErrMsg = SPS_ERR_INVALID_SIGNATURE;
            }
        }

        // Request is valid, go!
        if ( ! $epError ) {
            $response       = $data['MerchantApiResponse'];
            $request        = $response['PlatformRequest'];
            $transaction    = $response['PlatformTransaction'];
            // $user           = $response['PlatformAccount'];
            $sessionid      = $request['addinfo'] ?? null;
            // $request_id     = $request['request_id'];
            $order_id       = absint( $request['ref_id'] );
            // $transaction_id = $request['tr_id'];
            $order_key      = (!empty($sessionid)) ? wc_clean( $sessionid ) : null;
            $order          = wc_get_order( $order_id );
            $request_status = $request['status'];
        }

        if (empty($order) && !$epError) {
            $epError  = true;
            $epErrMsg = SPS_ERR_ORDER_NOT_FOUND;
        }

        // Get internal order and verify it hasn't already been processed
        if (! $epError) {
            //$this->log( "Purchase:\n". json_encode( $order ) );
            // Check if the order has already been processed
            if ( $request_status == 'refunded' ) {
                // IPN for refund
                $epDone = true;
                if ( $order->get_status() != 'refunded' ) {
                    $this->log( 'Order has been refunded' );
                    $processRefund = true;
                }
            } else if ( $order->get_status() === 'on-hold' ||  $order->get_status() === 'completed' ) {
                $this->log( 'Order has already been processed' );
                $epDone = true;
            }
        }

        // Check data against internal order
        if( ! $epError && ! $epDone ) {
            $this->log( 'Check data against internal order' );
            // Check order amount
            if ( ! $this->amounts_equal( floatval($request['amount']), $order->get_total() ) ) {
                $epError  = true;
                $epErrMsg = SPS_ERR_AMOUNT_MISMATCH;
            }
            // Check session ID
            elseif( strcasecmp( $sessionid, $order->get_order_key() ) != 0 ) {
                $epError  = true;
                $epErrMsg = SPS_ERR_SESSIONID_MISMATCH;
            }
        }

        // If an error occurred
        if ( $epError ) {
            if (!empty($epErrMsg))
                $this->log( 'Error occurred: '. $epErrMsg );

            if ( $this->send_debug_email ) {
                $this->log( 'Sending email notification' );
                // Send an email
                $subject = "SPS IPN error: ". ($epErrMsg ?? "");
                $body =
                    "Hi,\n\n".
                    "An SPS IPN on your website requires attention\n".
                    "------------------------------------------------------------\n".
                    "Site: ". $vendor_name ." (". $vendor_url .")\n".
                    "Remote IP Address: ".$_SERVER['REMOTE_ADDR']."\n".
                    "Remote host name: ". gethostbyaddr( $_SERVER['REMOTE_ADDR'] ) ."\n";
                if( isset( $request['ref_id'] ) )
                    $body .= "Reference: ". $request['ref_id'] ."\n";
                if( !empty($order) )
                    $body .= "Purchase ID: ". $order->get_id() ."\n";
                if( !empty($order) )
                    $body .= "User ID: ". $order->get_user_id() ."\n";
                if( isset( $response ) )
                    $body .= "Smart Processing Solution Response Status: ". $response['status'] ."\n";
                if( isset( $request ) )
                    $body .= "Smart Processing Solution Request ID: ". $request['request_id'] ."\n";
                if( isset( $request['status'] ) )
                    $body .= "Smart Processing Solution Request Status: ". $request['status'] ."\n";
                $body .=
                    "\nError: ". ($epErrMsg ?? "") ."\n";

                switch( ($epErrMsg ?? "") ) {
                    case SPS_ERR_AMOUNT_MISMATCH:
                        if( isset( $request ) && isset($order) )
                            $body .=
                                "Value received : ". $request['amount'] ."\n".
                                "Value should be: ". $order->get_formatted_order_total();
                    break;

                    case SPS_ERR_ORDER_ID_MISMATCH:
                        if( isset( $request ) && isset($order) )
                            $body .=
                                "Value received : ". $request['ref_id'] ."\n".
                                "Value should be: ". $order->get_id();
                    break;

                    case SPS_ERR_ORDER_INVALID:
                        if( isset( $request ) && isset($order) )
                            $body .=
                                "Value received : ". $request['addinfo'] ."\n".
                                "Value should be: ". $order->get_order_key();
                    break;

                    // For all other errors there is no need to add additional information
                    default:
                    break;
                }

                wp_mail( $epDebugEmail, $subject, $body );
            }
        } elseif ( ! $epDone ) {

            $this->log( 'Check status and update order' );

            if (empty($order) || empty($order_key) || empty($response) || empty($request)) {
                $this->log( 'Missing data, cannot continue' );
                exit;
            }

            if ( $order->get_order_key() !== $order_key ) {
                $this->log( 'Invalid data: order key mismatch, cannot continue' );
                exit;
            }

            switch ( strtolower( $response['status'] ) ) {
                case 'ok':
                    if ($request['status']=='completed') {
                        $this->log( '- Complete' );
                        $order->add_order_note( __( 'SPS payment completed', 'woocommerce-gateway-sps' ) );
                        $order->update_meta_data('_sps_request_id', $request['request_id']);
                        $order->update_meta_data('_sps_transaction_id', $request['tr_id']);
                        $order->save();
                        if ($this->complete_status=='on-hold') {
                            try {
                                $order->set_transaction_id($request['tr_id']);
                                $order->set_date_paid( time() );
                            } catch (WC_Data_Exception $e) {
                            }
                            $order->update_status('on-hold');
                        } else {
                            $order->payment_complete($request['tr_id']);
                        }
                    } else if ($request['status']=='refunded') {

                    }
                    if ( $this->send_debug_email ) {
                        $subject = "Smart Processing Solution IPN on your site";
                        $body =
                            "Hi,\n\n".
                            "A Smart Processing Solution transaction has been completed on your website\n".
                            "------------------------------------------------------------\n".
                            "Site: ". $vendor_name ." (". $vendor_url .")\n".
                            "Purchase ID: ". $order->get_id() ."\n".
                            "Smart Processing Solution Request ID: ". $request['request_id'] ."\n".
                            "Smart Processing Solution Transaction ID: ". $request['tr_id'] ."\n".
                            "Smart Processing Solution Payment Status: ". $request['status'] ."\n".
                            "Order Status Code: ". $order->get_status();
                        wp_mail( $epDebugEmail, $subject, $body );
                    }
                break;
                case 'ko':
                case 'error':
                    $this->log( '- Failed' );
                    $order->update_status( 'rejected', sprintf( __( 'Payment %s via IPN.', 'woocommerce-gateway-sps' ), strtolower( sanitize_text_field( $request['status'] ) ) ) );

                    if ( $this->send_debug_email ) {
                        $subject = "Smart Processing Solution IPN on your site";
                        $body =
                            "Hi,\n\n".
                            "A failed Smart Processing Solution transaction on your website requires attention\n".
                            "------------------------------------------------------------\n".
                            "Site: ". $vendor_name ." (". $vendor_url .")\n".
                            "Purchase ID: ". $order->get_id() ."\n".
                            "Smart Processing Solution Request ID: ". $request['request_id'] ."\n".
                            "Smart Processing Solution Transaction ID: ". $request['tr_id'] ."\n".
                            "Smart Processing Solution Payment Status: ". $request['status'] ."\n".
                            "Error/Message: ". $response['msg'] ."\n".
                            $response['info'] ."\n".
                            "Order Status Code: ". $order->get_status();
                        wp_mail( $epDebugEmail, $subject, $body );
                    }
                break;
                //case 'pending':
                //$this->log( '- Pending' );
                // Need to wait for "Completed" before processing
                // Don't do this: the same email is send for on-hold.processing, and complete which is not good!
                // $order->update_status( 'on-hold', sprintf( __( 'Payment %s via IPN.', 'woocommerce-gateway-sps' ), strtolower( sanitize_text_field( $request['status'] ) ) );

                //break;
                default:
                    // If unknown status, do nothing (safest course of action)
                    $order->add_order_note(sprintf( __( 'New Smart Processing Solution IPN received. IPN status: %s Request status: %s. %s', 'woocommerce-gateway-sps' ), $response['status'], $request['status'], $response['msg']) );
                break;
            }
        } else if ($processRefund) {
            $this->log( 'Check order' );

            if (empty($order) || empty($order_key) || empty($response) || empty($request)) {
                $this->log( 'Missing data, cannot continue' );
                exit;
            }

            if ( $order->get_order_key() !== $order_key ) {
                $this->log( 'Invalid data: order key mismatch, cannot continue' );
                exit;
            }

            if ($this->process_refund_ipn) {
                $order->add_order_note( __( 'Trying to process refund IPN', 'woocommerce-gateway-sps' ) );
                if (!empty($request['refund_amount']))
                    $refund_amount = $request['refund_amount'];
                else if (!empty($transaction['tr_amount_refunded']))
                    $refund_amount = $transaction['tr_amount_refunded'];
                else
                    $refund_amount = $request['amount'];
                $this->log( 'Refund amount: ' . $refund_amount );

                try {
                    $refund = wc_create_refund(array(
                        'amount' => $refund_amount,
                        'reason' => $request['status_infos'] ?? '',
                        'order_id' => $order->get_id(),
                        'refund_payment' => false
                    ));

                    if ($refund instanceof WP_Error) {
                        $order->add_order_note( __( 'Failed to create a refund:', 'woocommerce-gateway-sps' ) . ' ' . $refund->get_error_message() );
                        $this->log( 'Failed to create a refund: ' . $refund->get_error_message() );
                    } else {
                        $order->add_order_note( __( 'Payment refunded', 'woocommerce-gateway-sps' ) );
                        $this->log( 'Payment refunded with refund #' . $refund->get_id() );
                    }
                } catch (Exception $e) {
                    $epError  = true;
                    $epErrMsg = $e->getMessage();
                    $this->log( 'Failed to create a refund: ' . $epErrMsg );
                }
            } else {
                $order->add_order_note( __( 'Ignoring refund IPN', 'woocommerce-gateway-sps' ) );
            }

            if ( $this->send_debug_email ) {
                $subject = "Smart Processing Solution Refund IPN on your site";
                $body =
                    "Hi,\n\n".
                    "A Smart Processing Solution transaction has been refunded on your website\n".
                    "------------------------------------------------------------\n".
                    "Site: ". $vendor_name ." (". $vendor_url .")\n".
                    "Purchase ID: ". $order->get_id() ."\n".
                    "Refund ID: ". (isset($refund) && !($refund instanceof WP_Error) ? $refund->get_id() : "") ."\n".
                    "Smart Processing Solution Request ID: ". $request['request_id'] ."\n".
                    "Smart Processing Solution Transaction ID: ". $request['tr_id'] ."\n".
                    "Smart Processing Solution Payment Status: ". $request['status'] ."\n".
                    "Order Status Code: ". $order->get_status();
                wp_mail( $epDebugEmail, $subject, $body );
            }
        }

        return [
            'status' => $epError ? 'error' : 'ok',
            'code' => sanitize_title($epErrMsg), // Convert error message to slug format
            'message' => $epErrMsg,
            'order_id' => $order_id ?? null,
            'request_status' => $request_status ?? null,
            'platform' => array(
                'wp_version' => get_bloginfo('version'),
                'wc_version' => WC()->version,
                'plugin_version' => $this->version,
            )
        ];
    }

    /**
     * Setup constants.
     *
     * Setup common values and messages used by the Smart Processing Solution gateway.
     *
     * @since 1.1.0
     */
    public function setup_constants(): void
    {
        //// Create user agent string
        if (!defined('SPS_SOFTWARE_NAME'))
            define( 'SPS_SOFTWARE_NAME', 'WooCommerce' );
        if (!defined('SPS_SOFTWARE_VER'))
            define( 'SPS_SOFTWARE_VER', $this->get_woocommerce_version() );
        if (!defined('SPS_MODULE_NAME'))
            define( 'SPS_MODULE_NAME', 'WooCommerce-Gateway-SPS' );
        if (!defined('SPS_MODULE_VER'))
            define( 'SPS_MODULE_VER', $this->version );

        // Features
        // - PHP
        $epFeatures = 'PHP ' . phpversion() .';';

        // - cURL
        if ( in_array( 'curl', get_loaded_extensions() ) ) {
            if (!defined('SPS_CURL'))
                define( 'SPS_CURL', '' );
            $epVersion = curl_version();
            $epFeatures .= ' curl '. $epVersion['version'] .';';
        } else {
            $epFeatures .= ' nocurl;';
        }

        // Create user agrent
        if (!defined('SPS_USER_AGENT'))
            define( 'SPS_USER_AGENT', SPS_SOFTWARE_NAME .'/'. SPS_SOFTWARE_VER .' ('. trim( $epFeatures ) .') '. SPS_MODULE_NAME .'/'. SPS_MODULE_VER );

        // General Defines
        if (!defined('SPS_TIMEOUT'))
            define( 'SPS_TIMEOUT', 15 );
        if (!defined('SPS_EPSILON'))
            define( 'SPS_EPSILON', 0.01 );

        // Messages
        // Error
        if (!defined('SPS_ERR_ORDER_NOT_FOUND'))
            define( 'SPS_ERR_ORDER_NOT_FOUND', __( 'No order found with this reference', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_AMOUNT_MISMATCH'))
            define( 'SPS_ERR_AMOUNT_MISMATCH', __( 'Amount mismatch', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_BAD_ACCESS'))
            define( 'SPS_ERR_BAD_ACCESS', __( 'Bad access of page', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_BAD_SOURCE_IP'))
            define( 'SPS_ERR_BAD_SOURCE_IP', __( 'Bad source IP address', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_CONNECT_FAILED'))
            define( 'SPS_ERR_CONNECT_FAILED', __( 'Failed to connect to Smart Processing Solution', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_INVALID_SIGNATURE'))
            define( 'SPS_ERR_INVALID_SIGNATURE', __( 'Security signature mismatch', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_MERCHANT_ID_MISMATCH'))
            define( 'SPS_ERR_MERCHANT_ID_MISMATCH', __( 'Merchant ID mismatch', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_NO_SESSION'))
            define( 'SPS_ERR_NO_SESSION', __( 'No saved session found for IPN transaction', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_ORDER_ID_MISSING_URL'))
            define( 'SPS_ERR_ORDER_ID_MISSING_URL', __( 'Order ID not present in URL', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_ORDER_ID_MISMATCH'))
            define( 'SPS_ERR_ORDER_ID_MISMATCH', __( 'Order ID mismatch', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_ORDER_INVALID'))
            define( 'SPS_ERR_ORDER_INVALID', __( 'This order ID is invalid', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_ORDER_NUMBER_MISMATCH'))
            define( 'SPS_ERR_ORDER_NUMBER_MISMATCH', __( 'Order Number mismatch', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_ORDER_PROCESSED'))
            define( 'SPS_ERR_ORDER_PROCESSED', __( 'This order has already been processed', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_PDT_FAIL'))
            define( 'SPS_ERR_PDT_FAIL', __( 'PDT query failed', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_PDT_TOKEN_MISSING'))
            define( 'SPS_ERR_PDT_TOKEN_MISSING', __( 'PDT token not present in URL', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_SESSIONID_MISMATCH'))
            define( 'SPS_ERR_SESSIONID_MISMATCH', __( 'Session ID mismatch', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_ERR_UNKNOWN'))
            define( 'SPS_ERR_UNKNOWN', __( 'Unkown error occurred', 'woocommerce-gateway-sps' ) );

        // General
        if (!defined('SPS_MSG_OK'))
            define( 'SPS_MSG_OK', __( 'Payment was successful', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_MSG_FAILED'))
            define( 'SPS_MSG_FAILED', __( 'Payment has failed', 'woocommerce-gateway-sps' ) );
        if (!defined('SPS_MSG_PENDING'))
            define( 'SPS_MSG_PENDING',
                __( 'The payment is pending. Please note, you will receive another notification', 'woocommerce-gateway-sps' ).
                __( ' when the payment status changes to', 'woocommerce-gateway-sps' ).
                __( ' "Completed", or "Failed"', 'woocommerce-gateway-sps' ) );
    }

    /**
     * Log system processes in test mode.
     * @since 1.1.0
     */
    public function log( $message, $debug = false ): void
    {
        if ($debug) {
            // Verbose logging
            if ( 'yes' === $this->get_option( 'verbose_logging' ) )
                wc_get_logger()->debug( $message , ["source" => "sps"]);
            return;
        }
        wc_get_logger()->info( $message , ["source" => "sps"]);
    }

    function get_woocommerce_version() {
        return WC()->version;
    }

    /**
     * validate_signature()
     *
     * Validate the signature against the returned data.
     *
     * @param array $data
     * @return bool
     * @since 1.1.0
     */
    public function validate_signature (array $data ): bool
    {
        $response   = $data['MerchantApiResponse'];
        $request    = $response['PlatformRequest'];
        $string     = $request['request_id'].$request['amount'].$request['currency'].$request['status'].$this->merchant_key;
        $myhash     = md5($string);
        if ($myhash!=$response['hash']) {
            $this->log("Hash verification failed. MyHash: ".$myhash." Hash: ".$response['hash']);
            $this->log("Hash String: ".$string, true);
            return false;
        }
        return true;
    }

    /**
     * Get valid IP addresses from SPS API
     *
     * @return array|WP_Error Array of valid IP addresses or WP_Error on failure
     */
    private function get_valid_ips() {
        $response = wp_remote_get('https://www.smartprocessingsolution.com/api/processing/ip');

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (empty($data) || !isset($data['data']) || !is_array($data['data'])) {
            return new WP_Error(
                'sps_invalid_ip_response',
                __('Invalid response from SPS IP endpoint', 'woocommerce-gateway-sps')
            );
        }

        return $data['data'];
    }

    /**
     * Get latest version of the plugin
     *
     * @return string|WP_Error Array of valid IP addresses or WP_Error on failure
     */
    private function get_latest_version() {
        $response = wp_remote_get('https://www.smartprocessingsolution.com/api/processing/woo-plugin-version');

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (empty($data) || !isset($data['data']) || !is_string($data['data'])) {
            return new WP_Error(
                'sps_invalid_version_response',
                __('Invalid response from SPS version endpoint', 'woocommerce-gateway-sps')
            );
        }

        return $data['data']; //Expected format: 1.0.0
    }

    /**
     * Display admin notice if API credentials are missing
     */
    public function check_version(): void {
        // Only show notice if gateway is enabled
        if ('yes' !== $this->get_option('enabled')) {
            return;
        }

        $latest_version = $this->get_latest_version();
        if (is_wp_error($latest_version)) {
            $this->log('Error checking latest version: ' . $latest_version->get_error_message());
            return;
        }

        $current_version = $this->version;
        if (version_compare($current_version, $latest_version, '<')) {
            $this->log('New version available: ' . $latest_version);
            
            echo '<div class="notice notice-error is-dismissible">';
            echo '<p><strong>' . __('Smart Processing Solution Gateway', 'woocommerce-gateway-sps') . '</strong></p>';
            echo '<p>' . sprintf(
                /* translators: 1: Missing credentials list, 2: Settings URL */
                __('A new version of the Smart Processing Solution Gateway is available. Please <a href="%1$s">download the new version</a> and update the plugin.', 'woocommerce-gateway-sps'),
                esc_url("https://www.smartprocessingsolution.com/files/documents/gateway-smartprocessing-sps.zip")
            ) . '</p>';
            echo '</div>';
        }
    }

    /**
     * validate_ip()
     *
     * Validate the IP address to make sure it's coming from Smart Processing Solution.
     *
     * @param string $sourceIP
     * @return bool
     * @since 1.1.0
     */
    public function validate_ip( string $sourceIP ): bool
    {
        // Local IP (for dev and sites hosted on same servers than EP)
        if ( ! filter_var($sourceIP, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) )
            return true;
        $valid_ips = $this->get_valid_ips();   
        if (is_wp_error($valid_ips)) {
            error_log(sprintf(
                '[WC_SPS_Payments] Error fetching valid IPs: %s',
                $valid_ips->get_error_message()
            ));
            return false;
        }
        $this->log( "Valid IPs:\n". json_encode( $valid_ips ), true );
        $this->log( "Source IP:\n". $sourceIP, true );
        return in_array( $sourceIP, $valid_ips );
    }

    /**
     * validate_response_data()
     *
     * @param $post_data array Data received
     * @return bool
     * @since 1.1.0
     */
    public function validate_response_data(array $post_data ): bool
    {
        return !empty($post_data);
    }

    /**
     * amounts_equal()
     *
     * Checks to see whether the given amounts are equal using a proper floating
     * point comparison with an Epsilon which ensures that insignificant decimal
     * places are ignored in the comparison.
     *
     * eg: 100.00 is equal to 100.0001
     *
     * @param $amount1 float Float 1st amount for comparison
     * @param $amount2 float Float 2nd amount for comparison
     * @return bool
     * @since 1.1.0
     */
    public function amounts_equal (float $amount1, float $amount2 ): bool
    {
        return ! ( abs(  $amount1 -  $amount2 ) > SPS_EPSILON );
    }

    /**
     * Maybe hide the gateway for non-admin users
     * 
     * @param array $available_gateways Array of available payment gateways
     * @return array Modified array of available payment gateways
     */
    public function maybe_hide_gateway_for_non_admin($available_gateways) {
        if (!is_admin() && isset($available_gateways[$this->id])) {
            // Check if gateway should be hidden for non-admin users
            if ('yes' === $this->get_option('hide_for_non_admin_users') && !current_user_can('manage_options')) {
                unset($available_gateways[$this->id]);
            }
        }
        return $available_gateways;
    }

    /**
     * Process a refund if supported.
     *
     * @param int    $order_id Order ID
     * @param float  $amount   Refund amount
     * @param string $reason   Refund reason
     * @return bool|WP_Error True or false based on success, or a WP_Error object
     */
    public function process_refund($order_id, $amount = null, $reason = '') {
        $order = wc_get_order($order_id);
        if (!$order) {
            return new WP_Error('invalid_order', __('Invalid order ID', 'woocommerce-gateway-sps'));
        }

        // Check if refund with API is enabled
        if ('yes' !== $this->get_option('refund_with_api')) {
            return new WP_Error(
                'refund_api_disabled',
                __('Refund with API is disabled in gateway settings.', 'woocommerce-gateway-sps')
            );
        }

        // Check for required API credentials
        $api_key = $this->get_option('sps_api_key');
        $secret_key = $this->get_option('sps_secret_key');
        if (empty($api_key) || empty($secret_key)) {
            return new WP_Error(
                'missing_api_credentials',
                __('API credentials are required for refunds. Please configure them in the gateway settings.', 'woocommerce-gateway-sps')
            );
        }

        // Get the original transaction ID
        $transaction_id = $order->get_meta('_sps_request_id');
        if (empty($transaction_id)) {
            return new WP_Error(
                'missing_transaction_id',
                __('Original transaction ID not found.', 'woocommerce-gateway-sps')
            );
        }

        try {
            // Initialize API client
            $api_client = new WC_SPS_API_Client(
                $api_key,
                'yes' === $this->get_option('test_mode')
            );

            $this->log(sprintf(
                'Processing refund for order #%d. Amount: %s, Reason: %s, Transaction ID: %s',
                $order_id,
                $amount,
                $reason,
                $transaction_id
            ));

            // Call the refund API
            $response = $api_client->refund_request(
                $transaction_id,
                $secret_key,
                $reason,
                $amount,
                $order->get_currency()
            );

            if (is_wp_error($response)) {
                $this->log('Refund failed: ' . $response->get_error_message());
                return $response;
            }

            // Check response status
            if (!isset($response['status']) || $response['status'] !== 'ok') {
                $error_message = isset($response['msg']) ? $response['msg'] : __('Unknown error', 'woocommerce-gateway-sps');
                $this->log('Refund failed: ' . $error_message);
                return new WP_Error('refund_failed', $error_message);
            }

            // Add note to the order
            $order->add_order_note(
                sprintf(
                    __('Refund of %1$s initiated via API. Reason: %2$s', 'woocommerce-gateway-sps'),
                    wc_price($amount),
                    $reason
                )
            );

            $this->log('Refund processed successfully');
            return true;

        } catch (Exception $e) {
            $this->log('Refund failed with exception: ' . $e->getMessage());
            return new WP_Error('refund_error', $e->getMessage());
        }
    }

    /**
     * Display admin notice if API credentials are missing
     */
    public function check_api_credentials(): void {
        // Only show notice if gateway is enabled
        if ('yes' !== $this->get_option('enabled')) {
            return;
        }

        // Check if we're missing any credentials
        $missing_credentials = array();
        
        if (empty($this->merchant_id)) {
            $missing_credentials[] = __('Merchant ID', 'woocommerce-gateway-sps');
        }
        
        if (empty($this->merchant_key)) {
            $missing_credentials[] = __('Encryption / Hash Key', 'woocommerce-gateway-sps');
        }

        // Check API credentials
        if (empty($this->get_option('sps_api_key'))) {
            $missing_credentials[] = __('API Key', 'woocommerce-gateway-sps');
        }
        if (empty($this->get_option('sps_secret_key'))) {
            $missing_credentials[] = __('Secret Key', 'woocommerce-gateway-sps');
        }

        if (!empty($missing_credentials)) {
            $settings_url = admin_url('admin.php?page=wc-settings&tab=checkout&section=sps');
            
            echo '<div class="notice notice-error is-dismissible">';
            echo '<p><strong>' . __('Smart Processing Solution Gateway', 'woocommerce-gateway-sps') . '</strong></p>';
            echo '<p>' . sprintf(
                /* translators: 1: Missing credentials list, 2: Settings URL */
                __('The following credentials are missing: %1$s. Please <a href="%2$s">configure the plugin</a> to enable payments.', 'woocommerce-gateway-sps'),
                implode(', ', $missing_credentials),
                esc_url($settings_url)
            ) . '</p>';
            echo '</div>';
        }
    }

    /**
     * Get the current plugin settings
     * 
     * @return array Current settings with sensitive data masked
     */
    public function get_settings(): array {
        $settings = $this->settings;

        // Mask sensitive data
        $sensitive_fields = [
            'merchant_key',
            'sps_api_key',
            'sps_secret_key',
        ];

        foreach ($sensitive_fields as $field) {
            if (!empty($settings[$field])) {
                $settings[$field] = '••••••••' . substr($settings[$field], -4);
            }
        }

        // Add additional useful information
        $settings['notification_url'] = $this->response_url;
        $settings['platform'] = [
            'wp_version' => get_bloginfo('version'),
            'wc_version' => WC()->version,
            'plugin_version' => $this->version,
            'currency' => get_woocommerce_currency(),
            'test_mode' => 'yes' === $this->get_option('test_mode'),
            'verbose_logging' => 'yes' === $this->get_option('verbose_logging'),
        ];

        // Add validation settings
        $settings['validation'] = [
            'ip' => 'yes' === $this->get_option('validate_ipn_ip'),
            'signature' => 'yes' === $this->get_option('validate_ipn_signature'),
        ];

        // Add gateway status
        $settings['status'] = [
            'enabled' => 'yes' === $this->enabled,
            'ready' => $this->is_valid_for_use(),
            'available_currencies' => $this->available_currencies,
        ];

        return $settings;
    }

    /**
     * Get order data for API response
     * 
     * @param int $order_id Order ID
     * @return array|WP_Error Order data or error
     */
    public function get_order($order_id): array|WP_Error {
        $order = wc_get_order($order_id);
        
        if (!$order) {
            return new WP_Error(
                'order_not_found',
                __('Order not found', 'woocommerce-gateway-sps')
            );
        }

        // Basic order data
        $order_data = [
            'id' => $order->get_id(),
            'number' => $order->get_order_number(),
            'key' => $order->get_order_key(),
            'created_at' => $order->get_date_created()->format('c'),
            'updated_at' => $order->get_date_modified()->format('c'),
            'status' => $order->get_status(),
            'currency' => $order->get_currency(),
            'total' => $order->get_total(),
            'subtotal' => $order->get_subtotal(),
            'total_tax' => $order->get_total_tax(),
            'shipping_total' => $order->get_shipping_total(),
            'payment_method' => $order->get_payment_method(),
            'payment_method_title' => $order->get_payment_method_title(),
            'transaction_id' => $order->get_transaction_id(),
            'customer_note' => $order->get_customer_note(),
        ];

        // Customer data
        $order_data['customer'] = [
            'id' => $order->get_customer_id(),
            'ip_address' => $order->get_customer_ip_address(),
            'user_agent' => $order->get_customer_user_agent(),
            'billing' => [
                'first_name' => $order->get_billing_first_name(),
                'last_name' => $order->get_billing_last_name(),
                'company' => $order->get_billing_company(),
                'address_1' => $order->get_billing_address_1(),
                'address_2' => $order->get_billing_address_2(),
                'city' => $order->get_billing_city(),
                'state' => $order->get_billing_state(),
                'postcode' => $order->get_billing_postcode(),
                'country' => $order->get_billing_country(),
                'email' => $order->get_billing_email(),
                'phone' => $order->get_billing_phone(),
            ],
            'shipping' => [
                'first_name' => $order->get_shipping_first_name(),
                'last_name' => $order->get_shipping_last_name(),
                'company' => $order->get_shipping_company(),
                'address_1' => $order->get_shipping_address_1(),
                'address_2' => $order->get_shipping_address_2(),
                'city' => $order->get_shipping_city(),
                'state' => $order->get_shipping_state(),
                'postcode' => $order->get_shipping_postcode(),
                'country' => $order->get_shipping_country(),
            ],
        ];

        // Meta data
        $meta_data = [];
        foreach ($order->get_meta_data() as $meta) {
            // Only include SPS-related meta
            if (strpos($meta->key, '_sps_') === 0) {
                $meta_data[$meta->key] = $meta->value;
            }
        }
        if (!empty($meta_data)) {
            $order_data['meta'] = $meta_data;
        }

        // Add refunds if any
        $refunds = $order->get_refunds();
        if (!empty($refunds)) {
            $order_data['refunds'] = array_map(function($refund) {
                return [
                    'id' => $refund->get_id(),
                    'reason' => $refund->get_reason(),
                    'amount' => $refund->get_amount(),
                    'date_created' => $refund->get_date_created()->format('c'),
                ];
            }, $refunds);
        }

        return $order_data;
    }

    /**
     * Get the latest logs from the SPS gateway
     * 
     * @param int $limit Maximum number of log entries to return (default 100)
     * @return array Array of log entries
     */
    public function get_logs(int $limit = 100): array {
        try {
            $logger = wc_get_logger();
            $handler = new WC_Log_Handler_File();
            $files = $handler->get_log_files();
            
            // Look for the most recent sps log file
            $sps_logs = array_filter($files, function($file) {
                return strpos($file, 'sps-') === 0;
            });

            if (empty($sps_logs) || !defined('WC_LOG_DIR')) {
                return [
                    'status' => 'error',
                    'message' => 'No log files found'
                ];
            }

            // Sort files by date (newest first)
            usort($sps_logs, function($a, $b) {
                return filemtime(WC_LOG_DIR . $b) - filemtime(WC_LOG_DIR . $a);
            });

            // Get the most recent log file
            $latest_log = WC_LOG_DIR . reset($sps_logs);
            
            if (!file_exists($latest_log)) {
                return [
                    'status' => 'error',
                    'message' => 'Log file not found'
                ];
            }

            // Read the log file
            $logs = array_filter(
                array_map('trim', file($latest_log)),
                function($line) {
                    return !empty($line);
                }
            );

            // Get the last n entries
            $logs = array_slice($logs, -$limit);

            // Parse the logs into a structured format
            $formatted_logs = array_map(function($line) {
                $matches = [];
                if (preg_match('/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\+\d{2}:\d{2} ([A-Z]+) (.+)$/', $line, $matches)) {
                    return [
                        'timestamp' => substr($line, 0, 25),
                        'level' => $matches[1],
                        'message' => $matches[2]
                    ];
                }
                return [
                    'timestamp' => null,
                    'level' => 'UNKNOWN',
                    'message' => $line
                ];
            }, $logs);

            return [
                'status' => 'ok',
                'file' => basename($latest_log),
                'entries' => $formatted_logs
            ];

        } catch (Exception $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Verify payment status for an order using SPS API
     *
     * @param WC_Order $order
     * @return bool True if payment is verified, false otherwise
     */
    public function verify_payment_status($order) {
        if (!$order) {
            return false;
        }

        // Don't check if order is not pending payment
        if ($order->get_status() !== 'pending') {
            return false;
        }

        try {
            $api_client = new WC_SPS_API_Client(
                $this->get_option('sps_api_key'),
                'yes' === $this->get_option('test_mode')
            );

            // Search for requests with this order ID
            $search_result = $api_client->search_requests([
                'reference' => $order->get_id(),
                'status' => 'completed'
            ]);

            if (is_wp_error($search_result)) {
                $this->log('Payment verification failed: ' . $search_result->get_error_message());
                return false;
            }

            // If we find a completed payment, process it
            if (!empty($search_result['data']['requests'])) {
                foreach ($search_result['data']['requests'] as $request) {
                    // Verify amount matches
                    if ($this->amounts_equal($request['amount'], $order->get_total())) {
                        // Process the successful payment
                        if ($this->complete_status=='on-hold') {
                            try {
                                $order->set_transaction_id($request['request_id']);
                                $order->set_date_paid( time() );
                            } catch (WC_Data_Exception $e) {
                            }
                            $order->update_status('on-hold');
                        } else {
                            $order->payment_complete($request['request_id']);
                        }
                        $order->add_order_note(
                            sprintf(
                                __('Payment verified via API. Transaction ID: %s', 'woocommerce-gateway-sps'),
                                $request['request_id']
                            )
                        );
                        return true;
                    }
                }
            }

            return false;

        } catch (Exception $e) {
            $this->log('Payment verification exception: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Verify payment before order cancellation
     *
     * @param WC_Order $order
     */
    public function verify_before_cancel($order) {
        if ($order->get_payment_method() === $this->id) {
            $this->verify_payment_status($order);
        }
    }

    /**
     * Schedule a payment verification check
     *
     * @param int $order_id
     */
    public function schedule_payment_check($order_id) {
        $order = wc_get_order($order_id);
        if ($order && $order->get_payment_method() === $this->id) {
            // Schedule three verification attempts over 15 minutes
            wp_schedule_single_event(time() + 300, 'wc_sps_verify_payment', array($order));  // 5 minutes
            wp_schedule_single_event(time() + 600, 'wc_sps_verify_payment', array($order));  // 10 minutes
            wp_schedule_single_event(time() + 900, 'wc_sps_verify_payment', array($order));  // 15 minutes
        }
    }

    /**
     * Check payment status when customer returns to the site
     */
    public function check_return_payment_status() {
        if (!is_order_received_page()) {
            return;
        }

        $order_id = absint(get_query_var('order-received'));
        if (!$order_id) {
            return;
        }

        $order = wc_get_order($order_id);
        if ($order && $order->get_payment_method() === $this->id) {
            $this->verify_payment_status($order);
        }
    }

    /**
     * Add meta box to display SPS transactions
     */
    public function add_sps_transactions_meta_box() {
        add_meta_box(
            'wc_sps_transactions',
            __('SPS Transactions', 'woocommerce-gateway-sps'),
            array($this, 'render_sps_transactions_meta_box'),
            'shop_order',
            'normal',
            'default'
        );
    }

    /**
     * Render the SPS transactions meta box
     * 
     * @param WP_Post $post The post object
     */
    public function render_sps_transactions_meta_box($post) {
        $order = wc_get_order($post->ID);
        if (!$order) {
            return;
        }

        // Only show for SPS orders or if plugin is enabled
        if ($order->get_payment_method() !== $this->id && $this->enabled !== 'yes') {
            echo '<p>' . __('This order was not processed using Smart Processing Solution.', 'woocommerce-gateway-sps') . '</p>';
            return;
        }

        try {
            $api_client = new WC_SPS_API_Client(
                $this->get_option('sps_api_key'),
                'yes' === $this->get_option('test_mode')
            );

            // Search for all requests with this order ID
            $search_result = $api_client->search_requests([
                'reference' => $order->get_id()
            ]);

            if (is_wp_error($search_result)) {
                echo '<p class="error">' . esc_html($search_result->get_error_message()) . '</p>';
                return;
            }

            if (empty($search_result['data']['requests'])) {
                echo '<p>' . __('No SPS transactions found for this order.', 'woocommerce-gateway-sps') . '</p>';
                return;
            }

            echo '<table class="widefat">
                <thead>
                    <tr>
                        <th>' . __('Request ID', 'woocommerce-gateway-sps') . '</th>
                        <th>' . __('Transaction ID', 'woocommerce-gateway-sps') . '</th>
                        <th>' . __('Date', 'woocommerce-gateway-sps') . '</th>
                        <th>' . __('Amount', 'woocommerce-gateway-sps') . '</th>
                        <th>' . __('Status', 'woocommerce-gateway-sps') . '</th>
                    </tr>
                </thead>
                <tbody>';

            foreach ($search_result['data']['requests'] as $transaction) {
                $status_class = $this->get_status_class($transaction['status']);
                $amount = wc_price($transaction['amount'], ['currency' => $order->get_currency()]);
                $date = isset($transaction['date']) ? 
                    wp_date(get_option('date_format') . ' ' . get_option('time_format'), strtotime($transaction['date'])) : 
                    __('N/A', 'woocommerce-gateway-sps');
                
                $extention = 'yes' === $this->get_option('test_mode') ? 'biz' : 'com';
                $link = "https://www.smartprocessingsolution.{$extention}/merchants/myaccount/request/{$transaction['request_id']}";

                echo '<tr>
                    <td><a href="' . esc_url($link) . '" target="_blank" rel="noopener noreferrer">' . esc_html($transaction['request_id']) . '</a></td>
                    <td>' . esc_html($transaction['tr_id']) . '</td>
                    <td>' . esc_html($date) . '</td>
                    <td>' . $amount . '</td>
                    <td><mark class="order-status ' . esc_attr($status_class) . '"><span>' . esc_html($transaction['status']) . '</span></mark></td>
                </tr>';
            }

            echo '</tbody></table>';

        } catch (Exception $e) {
            echo '<p class="error">' . esc_html($e->getMessage()) . '</p>';
        }
    }

    /**
     * Get the status class for styling
     * 
     * @param string $status
     * @return string
     */
    private function get_status_class($status) {
        $status_classes = [
            'completed' => 'status-completed',
            'pending' => 'status-pending',
            'failed' => 'status-failed',
            'cancelled' => 'status-cancelled',
            'refunded' => 'status-refunded'
        ];

        return isset($status_classes[$status]) ? $status_classes[$status] : 'status-' . sanitize_html_class($status);
    }

    /**
     * Handle manual refund status changes
     *
     * @param int $order_id
     * @param WC_Order $order
     */
    public function handle_manual_refund($order_id, $order) {
        // Only process for our payment method
        if ($order->get_payment_method() !== $this->id) {
            return;
        }

        // Get the transaction ID that was stored when the payment was completed
        $transaction_id = $order->get_transaction_id();
        if (empty($transaction_id)) {
            $order->add_order_note(
                __('No SPS transaction ID found for this order.', 'woocommerce-gateway-sps')
            );
            return;
        }

        try {
            $api_client = new WC_SPS_API_Client(
                $this->get_option('sps_api_key'),
                'yes' === $this->get_option('test_mode')
            );

            // Get the specific transaction
            $result = $api_client->get_request($transaction_id);

            if (is_wp_error($result)) {
                $order->add_order_note(
                    sprintf(
                        __('Failed to retrieve SPS transaction %s: %s', 'woocommerce-gateway-sps'),
                        $transaction_id,
                        $result->get_error_message()
                    )
                );
                return;
            }

            // Check if we have a valid response
            if ($result['status'] !== 'OK' || empty($result['request'])) {
                $order->add_order_note(
                    sprintf(
                        __('Invalid response when retrieving SPS transaction %s', 'woocommerce-gateway-sps'),
                        $transaction_id
                    )
                );
                return;
            }

            $request = $result['request'];
            $transaction = $result['transaction'];

            // Check if transaction can be refunded
            if ($request['status'] !== 'completed' || !empty($request['refund_date'])) {
                $order->add_order_note(
                    sprintf(
                        __('SPS transaction %s cannot be refunded (status: %s, already refunded: %s)', 'woocommerce-gateway-sps'),
                        $transaction_id,
                        $request['status'],
                        !empty($request['refund_date']) ? 'yes' : 'no'
                    )
                );
                return;
            }

            // Process refund through SPS API
            $refund_result = $api_client->refund_request(
                $transaction_id,
                $this->get_option('sps_secret_key'),
                __('Manual refund from WooCommerce', 'woocommerce-gateway-sps'),
                $order->get_total(),
                $order->get_currency()
            );

            if (is_wp_error($refund_result)) {
                $order->add_order_note(
                    sprintf(
                        __('Failed to process SPS refund for transaction %s: %s', 'woocommerce-gateway-sps'),
                        $transaction_id,
                        $refund_result->get_error_message()
                    )
                );
                return;
            }

            // Check refund response
            if ($refund_result['status'] !== 'OK' || empty($refund_result['transaction'])) {
                $order->add_order_note(
                    sprintf(
                        __('Invalid response when processing SPS refund for transaction %s', 'woocommerce-gateway-sps'),
                        $transaction_id
                    )
                );
                return;
            }

            $refunded_transaction = $refund_result['transaction'];
            $refunded_request = $refund_result['request'];

            // Add detailed success note
            $order->add_order_note(
                sprintf(
                    __('SPS refund processed for transaction %s (Transaction ID: %s)', 'woocommerce-gateway-sps'),
                    $refunded_request['request_id'],
                    $refunded_transaction['tr_id']
                )
            );

        } catch (Exception $e) {
            $this->log('Refund processing error: ' . $e->getMessage());
            $order->add_order_note(
                sprintf(
                    __('Error processing SPS refund: %s', 'woocommerce-gateway-sps'),
                    $e->getMessage()
                )
            );
        }
    }
}
